# Deploying DataListCanada to Namecheap (PHP + React)

This guide explains how to deploy DataListCanada using PHP backend with React frontend to Namecheap shared hosting.

## Architecture

- **Frontend**: React (built as static files)
- **Backend**: PHP with PDO/SQLite
- **Database**: SQLite (file-based)
- **Hosting**: Namecheap shared hosting (cPanel)

## Prerequisites

1. Namecheap shared hosting account with cPanel access
2. Node.js installed locally (for building React frontend)
3. Domain name configured

## Build Steps

### 1. Build the React Frontend

Run this locally before uploading:

```bash
npm install
npm run build
```

This creates a `dist/public` folder with the compiled React app.

### 2. Prepare the Upload Package

Your upload should include:
- `api/` folder (PHP backend)
- `dist/public/` contents (React frontend)
- `data.db` (SQLite database, or it will be created)

## Deployment Structure

On your Namecheap hosting, create this structure:

```
public_html/
├── .htaccess          # Main routing (see below)
├── index.html         # React app entry point
├── assets/            # React app assets (JS, CSS)
├── api/               # PHP backend folder
│   ├── .htaccess      # API routing
│   ├── index.php      # API entry point
│   ├── config/
│   │   └── database.php
│   └── controllers/
│       ├── AuthController.php
│       ├── ProductController.php
│       ├── LeadController.php
│       ├── CreditController.php
│       ├── AdminController.php
│       ├── SettingsController.php
│       └── StripeController.php
└── data.db            # SQLite database (auto-created)
```

## Main .htaccess File

Create this `.htaccess` in `public_html/`:

```apache
RewriteEngine On

# Handle API requests - route to PHP
RewriteCond %{REQUEST_URI} ^/api/
RewriteRule ^api/(.*)$ api/index.php [QSA,L]

# Handle static assets
RewriteCond %{REQUEST_FILENAME} -f
RewriteRule ^ - [L]

# Handle React client-side routing
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^ index.html [L]
```

## Environment Variables

Set these in cPanel > Software > Select PHP Version > Options:

Or create a `.env` file in `api/` folder:

```
STRIPE_SECRET_KEY=sk_live_xxxxx
STRIPE_WEBHOOK_SECRET=whsec_xxxxx
APOLLO_API_KEY=xxxxx
SESSION_SECRET=your-random-secret-key
```

Then update `api/index.php` to load the env file:

```php
// At the top of index.php, before session_start()
if (file_exists(__DIR__ . '/.env')) {
    $lines = file(__DIR__ . '/.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            putenv(trim($key) . '=' . trim($value));
        }
    }
}
```

## Step-by-Step Deployment

### Step 1: Upload Files

1. Log into cPanel
2. Open File Manager
3. Navigate to `public_html`
4. Upload the `api/` folder
5. Upload contents of `dist/public/` to root of `public_html`
6. Create the main `.htaccess` file

### Step 2: Set Permissions

```
chmod 755 api/
chmod 644 api/*.php
chmod 644 api/controllers/*.php
chmod 644 api/config/*.php
chmod 666 data.db  # Writable for SQLite
```

### Step 3: Configure Environment

1. Go to cPanel > Software > Select PHP Version
2. Select PHP 8.0 or higher
3. Enable extensions: `pdo_sqlite`, `curl`, `json`, `session`

### Step 4: Set Up Stripe Webhook

1. In Stripe Dashboard, go to Developers > Webhooks
2. Add endpoint: `https://yourdomain.com/api/stripe/webhook`
3. Select event: `checkout.session.completed`
4. Copy the webhook secret to your environment

### Step 5: Create First Admin

1. Register an account at `https://yourdomain.com/register`
2. Access the database via cPanel > phpMyAdmin or download `data.db`
3. Run this SQL:
   ```sql
   UPDATE users SET role = 'admin' WHERE email = 'your@email.com';
   ```

## Testing

1. Visit `https://yourdomain.com` - should see React frontend
2. Visit `https://yourdomain.com/api/products` - should return JSON
3. Try logging in and accessing admin features

## Troubleshooting

### "500 Internal Server Error"
- Check PHP version (needs 8.0+)
- Check file permissions
- Check Apache error logs in cPanel

### "API Not Found"
- Verify `.htaccess` files are uploaded
- Check if mod_rewrite is enabled
- Verify API folder structure

### "Database Error"
- Ensure `data.db` has write permissions (666)
- Check if pdo_sqlite extension is enabled

### CORS Issues
- The API already includes CORS headers
- Make sure you're accessing from the same domain

## Database Backup

The SQLite database is stored in `data.db`. To backup:
1. Download `data.db` via File Manager
2. Store securely

## Security Notes

1. Keep `api/.env` file secure (not accessible from web)
2. Use HTTPS for all traffic
3. Regularly backup `data.db`
4. Keep PHP updated via cPanel
