<?php
class Database {
    private static $instance = null;
    private $db;
    private $dbPath;
    
    private function __construct() {
        $this->dbPath = __DIR__ . '/../../data.db';
        $this->connect();
        $this->createTables();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function connect() {
        try {
            $this->db = new PDO('sqlite:' . $this->dbPath);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->db->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            die(json_encode(['error' => 'Database connection failed: ' . $e->getMessage()]));
        }
    }
    
    public function getConnection() {
        return $this->db;
    }
    
    private function createTables() {
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS users (
                id TEXT PRIMARY KEY,
                email TEXT UNIQUE NOT NULL,
                password TEXT NOT NULL,
                first_name TEXT,
                last_name TEXT,
                profile_image_url TEXT,
                role TEXT NOT NULL DEFAULT 'user',
                credits INTEGER NOT NULL DEFAULT 0,
                created_at INTEGER,
                updated_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS credit_packages (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                description TEXT,
                credits INTEGER NOT NULL,
                price TEXT NOT NULL,
                price_per_credit TEXT,
                features TEXT NOT NULL DEFAULT '[]',
                is_popular INTEGER NOT NULL DEFAULT 0,
                button_text TEXT NOT NULL DEFAULT 'Get Started',
                sort_order INTEGER NOT NULL DEFAULT 0,
                is_active INTEGER NOT NULL DEFAULT 1,
                created_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS cold_leads (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                company_name TEXT NOT NULL,
                contact_person TEXT,
                email TEXT,
                phone TEXT,
                industry TEXT,
                website TEXT,
                address TEXT,
                city TEXT,
                state TEXT,
                postal_code TEXT,
                country TEXT,
                facebook_url TEXT,
                linkedin_url TEXT,
                twitter_url TEXT,
                google_my_business TEXT,
                notes TEXT,
                status TEXT NOT NULL DEFAULT 'new',
                assigned_to TEXT REFERENCES users(id),
                created_at INTEGER,
                updated_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS products (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                description TEXT NOT NULL,
                price TEXT NOT NULL,
                count INTEGER NOT NULL DEFAULT 0,
                tags TEXT NOT NULL DEFAULT '[]',
                type TEXT NOT NULL DEFAULT 'B2B',
                created_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS product_leads (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                product_id INTEGER NOT NULL REFERENCES products(id) ON DELETE CASCADE,
                lead_id INTEGER NOT NULL REFERENCES cold_leads(id) ON DELETE CASCADE
            );

            CREATE TABLE IF NOT EXISTS pending_purchases (
                id TEXT PRIMARY KEY,
                email TEXT NOT NULL,
                credits INTEGER NOT NULL,
                amount TEXT NOT NULL,
                stripe_session_id TEXT,
                created_at INTEGER,
                expires_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS saved_searches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id TEXT NOT NULL REFERENCES users(id),
                name TEXT NOT NULL,
                filters TEXT NOT NULL DEFAULT '{}',
                lead_count INTEGER NOT NULL DEFAULT 0,
                credit_cost INTEGER NOT NULL DEFAULT 0,
                created_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS purchased_searches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id TEXT NOT NULL REFERENCES users(id),
                saved_search_id INTEGER REFERENCES saved_searches(id),
                filters TEXT NOT NULL DEFAULT '{}',
                lead_count INTEGER NOT NULL,
                credits_spent INTEGER NOT NULL,
                purchased_at INTEGER
            );

            CREATE TABLE IF NOT EXISTS purchased_search_leads (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                purchased_search_id INTEGER NOT NULL REFERENCES purchased_searches(id) ON DELETE CASCADE,
                lead_id INTEGER NOT NULL REFERENCES cold_leads(id) ON DELETE CASCADE
            );

            CREATE TABLE IF NOT EXISTS site_settings (
                key TEXT PRIMARY KEY,
                value TEXT NOT NULL
            );
        ");
        
        // Create unique index for product_leads
        try {
            $this->db->exec("CREATE UNIQUE INDEX IF NOT EXISTS idx_product_leads_unique ON product_leads(product_id, lead_id);");
        } catch (PDOException $e) {
            // Index might already exist
        }
    }
}
