<?php
class AdminController {
    
    public static function uploadCsv() {
        AuthController::requireAdmin();
        
        if (!isset($_FILES['file'])) {
            http_response_code(400);
            echo json_encode(['error' => 'No file uploaded']);
            return;
        }
        
        $file = $_FILES['file'];
        $mapping = json_decode($_POST['mapping'] ?? '{}', true);
        $productId = $_POST['productId'] ?? null;
        
        if ($file['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'Upload failed']);
            return;
        }
        
        $content = file_get_contents($file['tmp_name']);
        $lines = explode("\n", $content);
        $headers = str_getcsv(array_shift($lines));
        
        $db = Database::getInstance()->getConnection();
        $imported = 0;
        $duplicates = 0;
        $newLeadIds = [];
        
        foreach ($lines as $line) {
            if (empty(trim($line))) continue;
            
            $row = str_getcsv($line);
            $data = [];
            
            foreach ($mapping as $dbField => $csvIndex) {
                if ($csvIndex !== null && isset($row[$csvIndex])) {
                    $data[$dbField] = $row[$csvIndex];
                }
            }
            
            if (empty($data['company_name'])) continue;
            
            // Check for duplicate
            $stmt = $db->prepare("SELECT id FROM cold_leads WHERE company_name = ? AND (email = ? OR (email IS NULL AND ? IS NULL))");
            $stmt->execute([$data['company_name'], $data['email'] ?? null, $data['email'] ?? null]);
            if ($stmt->fetch()) {
                $duplicates++;
                continue;
            }
            
            $now = time();
            $stmt = $db->prepare("
                INSERT INTO cold_leads (
                    company_name, contact_person, email, phone, industry, website,
                    address, city, state, postal_code, country,
                    facebook_url, linkedin_url, twitter_url, google_my_business,
                    notes, status, created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'new', ?, ?)
            ");
            $stmt->execute([
                $data['company_name'],
                $data['contact_person'] ?? null,
                $data['email'] ?? null,
                $data['phone'] ?? null,
                $data['industry'] ?? null,
                $data['website'] ?? null,
                $data['address'] ?? null,
                $data['city'] ?? null,
                $data['state'] ?? null,
                $data['postal_code'] ?? null,
                $data['country'] ?? 'Canada',
                $data['facebook_url'] ?? null,
                $data['linkedin_url'] ?? null,
                $data['twitter_url'] ?? null,
                $data['google_my_business'] ?? null,
                $data['notes'] ?? null,
                $now,
                $now
            ]);
            
            $newLeadIds[] = $db->lastInsertId();
            $imported++;
        }
        
        // Add to product if specified
        if ($productId && !empty($newLeadIds)) {
            foreach ($newLeadIds as $leadId) {
                try {
                    $stmt = $db->prepare("INSERT OR IGNORE INTO product_leads (product_id, lead_id) VALUES (?, ?)");
                    $stmt->execute([$productId, $leadId]);
                } catch (Exception $e) {}
            }
            
            $stmt = $db->prepare("UPDATE products SET count = (SELECT COUNT(*) FROM product_leads WHERE product_id = ?) WHERE id = ?");
            $stmt->execute([$productId, $productId]);
        }
        
        echo json_encode([
            'imported' => $imported,
            'duplicates' => $duplicates,
            'leadIds' => $newLeadIds
        ]);
    }
    
    public static function getUsers() {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->query("SELECT id, email, first_name, last_name, role, credits, created_at FROM users ORDER BY created_at DESC");
        $users = $stmt->fetchAll();
        
        $result = array_map(function($u) {
            return [
                'id' => $u['id'],
                'email' => $u['email'],
                'firstName' => $u['first_name'],
                'lastName' => $u['last_name'],
                'role' => $u['role'],
                'credits' => $u['credits'],
                'createdAt' => $u['created_at'] ? date('c', $u['created_at']) : null
            ];
        }, $users);
        
        echo json_encode($result);
    }
    
    public static function updateUser($id) {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $updates = [];
        $params = [];
        
        if (isset($data['role'])) {
            $updates[] = "role = ?";
            $params[] = $data['role'];
        }
        if (isset($data['credits'])) {
            $updates[] = "credits = ?";
            $params[] = $data['credits'];
        }
        if (isset($data['firstName'])) {
            $updates[] = "first_name = ?";
            $params[] = $data['firstName'];
        }
        if (isset($data['lastName'])) {
            $updates[] = "last_name = ?";
            $params[] = $data['lastName'];
        }
        
        if (empty($updates)) {
            echo json_encode(['success' => true]);
            return;
        }
        
        $params[] = $id;
        $stmt = $db->prepare("UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?");
        $stmt->execute($params);
        
        echo json_encode(['success' => true]);
    }
    
    public static function getStats() {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        
        $totalLeads = $db->query("SELECT COUNT(*) FROM cold_leads")->fetchColumn();
        $totalProducts = $db->query("SELECT COUNT(*) FROM products")->fetchColumn();
        $totalUsers = $db->query("SELECT COUNT(*) FROM users")->fetchColumn();
        $totalCredits = $db->query("SELECT SUM(credits) FROM users")->fetchColumn() ?? 0;
        
        echo json_encode([
            'totalLeads' => (int)$totalLeads,
            'totalProducts' => (int)$totalProducts,
            'totalUsers' => (int)$totalUsers,
            'totalCredits' => (int)$totalCredits
        ]);
    }
    
    public static function enrichLead() {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $leadId = $data['leadId'] ?? null;
        
        if (!$leadId) {
            http_response_code(400);
            echo json_encode(['error' => 'Lead ID required']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM cold_leads WHERE id = ?");
        $stmt->execute([$leadId]);
        $lead = $stmt->fetch();
        
        if (!$lead) {
            http_response_code(404);
            echo json_encode(['error' => 'Lead not found']);
            return;
        }
        
        // Try web scraping if website exists
        $enrichedData = [];
        
        if (!empty($lead['website'])) {
            $enrichedData = self::scrapeWebsite($lead['website']);
        }
        
        // Try Apollo.io if API key exists
        $apolloApiKey = getenv('APOLLO_API_KEY');
        if ($apolloApiKey && !empty($lead['company_name'])) {
            $apolloData = self::apolloEnrich($lead, $apolloApiKey);
            $enrichedData = array_merge($enrichedData, $apolloData);
        }
        
        // Update lead with enriched data
        if (!empty($enrichedData)) {
            $updates = [];
            $params = [];
            
            foreach ($enrichedData as $field => $value) {
                if ($value && empty($lead[$field])) {
                    $updates[] = "$field = ?";
                    $params[] = $value;
                }
            }
            
            if (!empty($updates)) {
                $params[] = $leadId;
                $stmt = $db->prepare("UPDATE cold_leads SET " . implode(', ', $updates) . ", updated_at = " . time() . " WHERE id = ?");
                $stmt->execute($params);
            }
        }
        
        // Return updated lead
        $stmt = $db->prepare("SELECT * FROM cold_leads WHERE id = ?");
        $stmt->execute([$leadId]);
        echo json_encode($stmt->fetch());
    }
    
    private static function scrapeWebsite($url) {
        $data = [];
        
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'user_agent' => 'Mozilla/5.0 (compatible; DataListBot/1.0)'
                ]
            ]);
            
            $html = @file_get_contents($url, false, $context);
            if (!$html) return $data;
            
            // Extract social links
            if (preg_match('/facebook\.com\/[a-zA-Z0-9._-]+/i', $html, $matches)) {
                $data['facebook_url'] = 'https://www.' . $matches[0];
            }
            if (preg_match('/linkedin\.com\/company\/[a-zA-Z0-9._-]+/i', $html, $matches)) {
                $data['linkedin_url'] = 'https://www.' . $matches[0];
            }
            if (preg_match('/twitter\.com\/[a-zA-Z0-9_]+/i', $html, $matches)) {
                $data['twitter_url'] = 'https://' . $matches[0];
            }
            
            // Extract email
            if (preg_match('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/i', $html, $matches)) {
                $email = $matches[0];
                if (!preg_match('/\.(png|jpg|gif|css|js)$/i', $email)) {
                    $data['email'] = $email;
                }
            }
            
            // Extract phone
            if (preg_match('/(\+?1[-.\s]?)?(\(?\d{3}\)?[-.\s]?)?\d{3}[-.\s]?\d{4}/', $html, $matches)) {
                $data['phone'] = $matches[0];
            }
            
        } catch (Exception $e) {
            // Scraping failed, continue
        }
        
        return $data;
    }
    
    private static function apolloEnrich($lead, $apiKey) {
        $data = [];
        
        try {
            $postData = json_encode([
                'organization_name' => $lead['company_name'],
                'domain' => parse_url($lead['website'] ?? '', PHP_URL_HOST)
            ]);
            
            $ch = curl_init('https://api.apollo.io/v1/organizations/enrich');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Cache-Control: no-cache',
                'X-Api-Key: ' . $apiKey
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            
            $response = curl_exec($ch);
            curl_close($ch);
            
            if ($response) {
                $result = json_decode($response, true);
                $org = $result['organization'] ?? [];
                
                if (!empty($org['linkedin_url'])) $data['linkedin_url'] = $org['linkedin_url'];
                if (!empty($org['twitter_url'])) $data['twitter_url'] = $org['twitter_url'];
                if (!empty($org['facebook_url'])) $data['facebook_url'] = $org['facebook_url'];
                if (!empty($org['phone'])) $data['phone'] = $org['phone'];
                if (!empty($org['industry'])) $data['industry'] = $org['industry'];
            }
        } catch (Exception $e) {
            // Apollo enrichment failed
        }
        
        return $data;
    }
    
    public static function apolloStatus() {
        AuthController::requireAdmin();
        
        $apiKey = getenv('APOLLO_API_KEY');
        echo json_encode([
            'configured' => !empty($apiKey),
            'hasCredits' => true
        ]);
    }
    
    public static function apolloSearch() {
        AuthController::requireAdmin();
        
        $apiKey = getenv('APOLLO_API_KEY');
        if (!$apiKey) {
            http_response_code(400);
            echo json_encode(['error' => 'Apollo API key not configured']);
            return;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        $postData = json_encode([
            'q_organization_keyword_tags' => $data['keywords'] ?? [],
            'organization_locations' => $data['locations'] ?? ['Canada'],
            'per_page' => $data['perPage'] ?? 25,
            'page' => $data['page'] ?? 1
        ]);
        
        $ch = curl_init('https://api.apollo.io/v1/mixed_companies/search');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Cache-Control: no-cache',
            'X-Api-Key: ' . $apiKey
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            http_response_code($httpCode);
            echo json_encode(['error' => 'Apollo API request failed']);
            return;
        }
        
        echo $response;
    }
}
