<?php
class AuthController {
    
    public static function register() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['email']) || empty($data['password'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Email and password are required']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        
        // Check if email exists
        $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$data['email']]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Email already registered']);
            return;
        }
        
        $id = bin2hex(random_bytes(16));
        $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
        $now = time();
        
        // Check for pending purchase
        $pendingCredits = 0;
        if (!empty($data['pendingPurchaseId'])) {
            $stmt = $db->prepare("SELECT credits FROM pending_purchases WHERE id = ? AND email = ?");
            $stmt->execute([$data['pendingPurchaseId'], $data['email']]);
            $pending = $stmt->fetch();
            if ($pending) {
                $pendingCredits = $pending['credits'];
                $stmt = $db->prepare("DELETE FROM pending_purchases WHERE id = ?");
                $stmt->execute([$data['pendingPurchaseId']]);
            }
        }
        
        $stmt = $db->prepare("
            INSERT INTO users (id, email, password, first_name, last_name, role, credits, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, 'user', ?, ?, ?)
        ");
        $stmt->execute([
            $id,
            $data['email'],
            $hashedPassword,
            $data['firstName'] ?? null,
            $data['lastName'] ?? null,
            $pendingCredits,
            $now,
            $now
        ]);
        
        session_regenerate_id(true);
        $_SESSION['user_id'] = $id;
        
        $stmt = $db->prepare("SELECT id, email, first_name, last_name, role, credits FROM users WHERE id = ?");
        $stmt->execute([$id]);
        $user = $stmt->fetch();
        
        echo json_encode([
            'id' => $user['id'],
            'email' => $user['email'],
            'firstName' => $user['first_name'],
            'lastName' => $user['last_name'],
            'role' => $user['role'],
            'credits' => $user['credits']
        ]);
    }
    
    public static function login() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['email']) || empty($data['password'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Email and password are required']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$data['email']]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($data['password'], $user['password'])) {
            http_response_code(401);
            echo json_encode(['error' => 'Invalid email or password']);
            return;
        }
        
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['id'];
        
        echo json_encode([
            'id' => $user['id'],
            'email' => $user['email'],
            'firstName' => $user['first_name'],
            'lastName' => $user['last_name'],
            'role' => $user['role'],
            'credits' => $user['credits']
        ]);
    }
    
    public static function logout() {
        session_destroy();
        echo json_encode(['success' => true]);
    }
    
    public static function getUser() {
        if (empty($_SESSION['user_id'])) {
            http_response_code(401);
            echo json_encode(['error' => 'Not authenticated']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT id, email, first_name, last_name, role, credits FROM users WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch();
        
        if (!$user) {
            http_response_code(401);
            echo json_encode(['error' => 'User not found']);
            return;
        }
        
        echo json_encode([
            'id' => $user['id'],
            'email' => $user['email'],
            'firstName' => $user['first_name'],
            'lastName' => $user['last_name'],
            'role' => $user['role'],
            'credits' => $user['credits']
        ]);
    }
    
    public static function requireAuth() {
        if (empty($_SESSION['user_id'])) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            exit;
        }
        return $_SESSION['user_id'];
    }
    
    public static function requireAdmin() {
        $userId = self::requireAuth();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT role FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user || $user['role'] !== 'admin') {
            http_response_code(403);
            echo json_encode(['error' => 'Admin access required']);
            exit;
        }
        
        return $userId;
    }
}
