<?php
class LeadController {
    
    public static function getAll() {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = ($page - 1) * $limit;
        $search = $_GET['search'] ?? '';
        
        $where = "1=1";
        $params = [];
        
        if ($search) {
            $where .= " AND (company_name LIKE ? OR contact_person LIKE ? OR email LIKE ?)";
            $searchParam = "%$search%";
            $params = [$searchParam, $searchParam, $searchParam];
        }
        
        $stmt = $db->prepare("SELECT * FROM cold_leads WHERE $where ORDER BY id DESC LIMIT ? OFFSET ?");
        $stmt->execute(array_merge($params, [$limit, $offset]));
        $leads = $stmt->fetchAll();
        
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM cold_leads WHERE $where");
        $stmt->execute($params);
        $total = $stmt->fetch()['total'];
        
        echo json_encode([
            'leads' => $leads,
            'total' => $total,
            'page' => $page,
            'totalPages' => ceil($total / $limit)
        ]);
    }
    
    public static function getOne($id) {
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM cold_leads WHERE id = ?");
        $stmt->execute([$id]);
        $lead = $stmt->fetch();
        
        if (!$lead) {
            http_response_code(404);
            echo json_encode(['error' => 'Lead not found']);
            return;
        }
        
        echo json_encode($lead);
    }
    
    public static function create() {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        $now = time();
        
        $stmt = $db->prepare("
            INSERT INTO cold_leads (
                company_name, contact_person, email, phone, industry, website,
                address, city, state, postal_code, country,
                facebook_url, linkedin_url, twitter_url, google_my_business,
                notes, status, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'new', ?, ?)
        ");
        $stmt->execute([
            $data['company_name'] ?? $data['companyName'] ?? '',
            $data['contact_person'] ?? $data['contactPerson'] ?? null,
            $data['email'] ?? null,
            $data['phone'] ?? null,
            $data['industry'] ?? null,
            $data['website'] ?? null,
            $data['address'] ?? null,
            $data['city'] ?? null,
            $data['state'] ?? null,
            $data['postal_code'] ?? $data['postalCode'] ?? null,
            $data['country'] ?? 'Canada',
            $data['facebook_url'] ?? $data['facebookUrl'] ?? null,
            $data['linkedin_url'] ?? $data['linkedinUrl'] ?? null,
            $data['twitter_url'] ?? $data['twitterUrl'] ?? null,
            $data['google_my_business'] ?? $data['googleMyBusiness'] ?? null,
            $data['notes'] ?? null,
            $now,
            $now
        ]);
        
        $id = $db->lastInsertId();
        self::getOne($id);
    }
    
    public static function update($id) {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            UPDATE cold_leads SET
                company_name = ?, contact_person = ?, email = ?, phone = ?, industry = ?,
                website = ?, address = ?, city = ?, state = ?, postal_code = ?, country = ?,
                facebook_url = ?, linkedin_url = ?, twitter_url = ?, google_my_business = ?,
                notes = ?, updated_at = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $data['company_name'] ?? $data['companyName'] ?? '',
            $data['contact_person'] ?? $data['contactPerson'] ?? null,
            $data['email'] ?? null,
            $data['phone'] ?? null,
            $data['industry'] ?? null,
            $data['website'] ?? null,
            $data['address'] ?? null,
            $data['city'] ?? null,
            $data['state'] ?? null,
            $data['postal_code'] ?? $data['postalCode'] ?? null,
            $data['country'] ?? 'Canada',
            $data['facebook_url'] ?? $data['facebookUrl'] ?? null,
            $data['linkedin_url'] ?? $data['linkedinUrl'] ?? null,
            $data['twitter_url'] ?? $data['twitterUrl'] ?? null,
            $data['google_my_business'] ?? $data['googleMyBusiness'] ?? null,
            $data['notes'] ?? null,
            time(),
            $id
        ]);
        
        self::getOne($id);
    }
    
    public static function delete($id) {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("DELETE FROM cold_leads WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['success' => true]);
    }
    
    public static function bulkDelete() {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $ids = $data['ids'] ?? [];
        
        if (empty($ids)) {
            echo json_encode(['deleted' => 0]);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        
        $stmt = $db->prepare("DELETE FROM cold_leads WHERE id IN ($placeholders)");
        $stmt->execute($ids);
        
        echo json_encode(['deleted' => $stmt->rowCount()]);
    }
    
    public static function getProducts($leadId) {
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            SELECT p.* FROM products p
            JOIN product_leads pl ON p.id = pl.product_id
            WHERE pl.lead_id = ?
        ");
        $stmt->execute([$leadId]);
        $products = $stmt->fetchAll();
        
        echo json_encode($products);
    }
    
    public static function getFilterOptions() {
        $db = Database::getInstance()->getConnection();
        
        $industries = $db->query("SELECT DISTINCT industry FROM cold_leads WHERE industry IS NOT NULL AND industry != '' ORDER BY industry")->fetchAll(PDO::FETCH_COLUMN);
        $cities = $db->query("SELECT DISTINCT city FROM cold_leads WHERE city IS NOT NULL AND city != '' ORDER BY city")->fetchAll(PDO::FETCH_COLUMN);
        $states = $db->query("SELECT DISTINCT state FROM cold_leads WHERE state IS NOT NULL AND state != '' ORDER BY state")->fetchAll(PDO::FETCH_COLUMN);
        
        echo json_encode([
            'industries' => $industries,
            'cities' => $cities,
            'provinces' => $states
        ]);
    }
    
    public static function search() {
        $userId = AuthController::requireAuth();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $where = "1=1";
        $params = [];
        
        if (!empty($data['industries'])) {
            $placeholders = implode(',', array_fill(0, count($data['industries']), '?'));
            $where .= " AND industry IN ($placeholders)";
            $params = array_merge($params, $data['industries']);
        }
        
        if (!empty($data['cities'])) {
            $placeholders = implode(',', array_fill(0, count($data['cities']), '?'));
            $where .= " AND city IN ($placeholders)";
            $params = array_merge($params, $data['cities']);
        }
        
        if (!empty($data['provinces'])) {
            $placeholders = implode(',', array_fill(0, count($data['provinces']), '?'));
            $where .= " AND state IN ($placeholders)";
            $params = array_merge($params, $data['provinces']);
        }
        
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM cold_leads WHERE $where");
        $stmt->execute($params);
        $count = $stmt->fetch()['count'];
        
        // Get credit cost from settings
        $stmt = $db->prepare("SELECT value FROM site_settings WHERE key = 'creditCostPerLead'");
        $stmt->execute();
        $setting = $stmt->fetch();
        $costPerLead = $setting ? (float)$setting['value'] : 1;
        
        echo json_encode([
            'count' => $count,
            'creditCost' => ceil($count * $costPerLead)
        ]);
    }
    
    public static function getSavedSearches() {
        $userId = AuthController::requireAuth();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM saved_searches WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        $searches = $stmt->fetchAll();
        
        $result = array_map(function($s) {
            return [
                'id' => $s['id'],
                'userId' => $s['user_id'],
                'name' => $s['name'],
                'filters' => json_decode($s['filters'], true),
                'leadCount' => $s['lead_count'],
                'creditCost' => $s['credit_cost'],
                'createdAt' => date('c', $s['created_at'])
            ];
        }, $searches);
        
        echo json_encode($result);
    }
    
    public static function saveSearch() {
        $userId = AuthController::requireAuth();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            INSERT INTO saved_searches (user_id, name, filters, lead_count, credit_cost, created_at)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $userId,
            $data['name'],
            json_encode($data['filters'] ?? []),
            $data['leadCount'] ?? 0,
            $data['creditCost'] ?? 0,
            time()
        ]);
        
        echo json_encode(['id' => $db->lastInsertId()]);
    }
    
    public static function deleteSavedSearch($id) {
        $userId = AuthController::requireAuth();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("DELETE FROM saved_searches WHERE id = ? AND user_id = ?");
        $stmt->execute([$id, $userId]);
        
        echo json_encode(['success' => true]);
    }
    
    public static function getPurchasedSearches() {
        $userId = AuthController::requireAuth();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM purchased_searches WHERE user_id = ? ORDER BY purchased_at DESC");
        $stmt->execute([$userId]);
        $searches = $stmt->fetchAll();
        
        $result = array_map(function($s) {
            return [
                'id' => $s['id'],
                'userId' => $s['user_id'],
                'savedSearchId' => $s['saved_search_id'],
                'filters' => json_decode($s['filters'], true),
                'leadCount' => $s['lead_count'],
                'creditsSpent' => $s['credits_spent'],
                'purchasedAt' => date('c', $s['purchased_at'])
            ];
        }, $searches);
        
        echo json_encode($result);
    }
    
    public static function purchaseSearch() {
        $userId = AuthController::requireAuth();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        // Get user credits
        $stmt = $db->prepare("SELECT credits FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if ($user['credits'] < $data['creditCost']) {
            http_response_code(400);
            echo json_encode(['error' => 'Insufficient credits']);
            return;
        }
        
        // Build query to get leads
        $where = "1=1";
        $params = [];
        $filters = $data['filters'] ?? [];
        
        if (!empty($filters['industries'])) {
            $placeholders = implode(',', array_fill(0, count($filters['industries']), '?'));
            $where .= " AND industry IN ($placeholders)";
            $params = array_merge($params, $filters['industries']);
        }
        
        if (!empty($filters['cities'])) {
            $placeholders = implode(',', array_fill(0, count($filters['cities']), '?'));
            $where .= " AND city IN ($placeholders)";
            $params = array_merge($params, $filters['cities']);
        }
        
        if (!empty($filters['provinces'])) {
            $placeholders = implode(',', array_fill(0, count($filters['provinces']), '?'));
            $where .= " AND state IN ($placeholders)";
            $params = array_merge($params, $filters['provinces']);
        }
        
        $stmt = $db->prepare("SELECT id FROM cold_leads WHERE $where");
        $stmt->execute($params);
        $leadIds = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Deduct credits
        $stmt = $db->prepare("UPDATE users SET credits = credits - ? WHERE id = ?");
        $stmt->execute([$data['creditCost'], $userId]);
        
        // Create purchased search
        $stmt = $db->prepare("
            INSERT INTO purchased_searches (user_id, saved_search_id, filters, lead_count, credits_spent, purchased_at)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $userId,
            $data['savedSearchId'] ?? null,
            json_encode($filters),
            count($leadIds),
            $data['creditCost'],
            time()
        ]);
        $purchasedSearchId = $db->lastInsertId();
        
        // Link leads to purchased search
        foreach ($leadIds as $leadId) {
            $stmt = $db->prepare("INSERT INTO purchased_search_leads (purchased_search_id, lead_id) VALUES (?, ?)");
            $stmt->execute([$purchasedSearchId, $leadId]);
        }
        
        // Delete saved search if provided
        if (!empty($data['savedSearchId'])) {
            $stmt = $db->prepare("DELETE FROM saved_searches WHERE id = ? AND user_id = ?");
            $stmt->execute([$data['savedSearchId'], $userId]);
        }
        
        echo json_encode(['id' => $purchasedSearchId, 'leadCount' => count($leadIds)]);
    }
    
    public static function exportPurchasedSearch($id) {
        $userId = AuthController::requireAuth();
        
        $db = Database::getInstance()->getConnection();
        
        // Verify ownership
        $stmt = $db->prepare("SELECT * FROM purchased_searches WHERE id = ? AND user_id = ?");
        $stmt->execute([$id, $userId]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Search not found']);
            return;
        }
        
        // Get leads
        $stmt = $db->prepare("
            SELECT cl.* FROM cold_leads cl
            JOIN purchased_search_leads psl ON cl.id = psl.lead_id
            WHERE psl.purchased_search_id = ?
        ");
        $stmt->execute([$id]);
        $leads = $stmt->fetchAll();
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="leads-export.csv"');
        
        $output = fopen('php://output', 'w');
        
        if (count($leads) > 0) {
            fputcsv($output, array_keys($leads[0]));
            foreach ($leads as $lead) {
                fputcsv($output, $lead);
            }
        }
        
        fclose($output);
        exit;
    }
}
