<?php
class ProductController {
    
    public static function getAll() {
        $db = Database::getInstance()->getConnection();
        $stmt = $db->query("SELECT * FROM products ORDER BY created_at DESC");
        $products = $stmt->fetchAll();
        
        $result = array_map(function($p) {
            return [
                'id' => $p['id'],
                'title' => $p['title'],
                'description' => $p['description'],
                'price' => $p['price'],
                'count' => $p['count'],
                'tags' => json_decode($p['tags'] ?? '[]', true),
                'type' => $p['type'],
                'createdAt' => date('c', $p['created_at'])
            ];
        }, $products);
        
        echo json_encode($result);
    }
    
    public static function getOne($id) {
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM products WHERE id = ?");
        $stmt->execute([$id]);
        $p = $stmt->fetch();
        
        if (!$p) {
            http_response_code(404);
            echo json_encode(['error' => 'Product not found']);
            return;
        }
        
        echo json_encode([
            'id' => $p['id'],
            'title' => $p['title'],
            'description' => $p['description'],
            'price' => $p['price'],
            'count' => $p['count'],
            'tags' => json_decode($p['tags'] ?? '[]', true),
            'type' => $p['type'],
            'createdAt' => date('c', $p['created_at'])
        ]);
    }
    
    public static function create() {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            INSERT INTO products (title, description, price, count, tags, type, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $data['title'] ?? '',
            $data['description'] ?? '',
            $data['price'] ?? '0',
            0,
            json_encode($data['tags'] ?? []),
            $data['type'] ?? 'B2B',
            time()
        ]);
        
        $id = $db->lastInsertId();
        self::getOne($id);
    }
    
    public static function update($id) {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $db = Database::getInstance()->getConnection();
        
        $stmt = $db->prepare("
            UPDATE products SET title = ?, description = ?, price = ?, tags = ?, type = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $data['title'],
            $data['description'] ?? '',
            $data['price'],
            json_encode($data['tags'] ?? []),
            $data['type'] ?? 'B2B',
            $id
        ]);
        
        self::getOne($id);
    }
    
    public static function delete($id) {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("DELETE FROM products WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['success' => true]);
    }
    
    public static function getLeads($productId) {
        AuthController::requireAdmin();
        
        $db = Database::getInstance()->getConnection();
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = ($page - 1) * $limit;
        
        $stmt = $db->prepare("
            SELECT cl.* FROM cold_leads cl
            JOIN product_leads pl ON cl.id = pl.lead_id
            WHERE pl.product_id = ?
            ORDER BY cl.id DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$productId, $limit, $offset]);
        $leads = $stmt->fetchAll();
        
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM product_leads WHERE product_id = ?");
        $stmt->execute([$productId]);
        $total = $stmt->fetch()['total'];
        
        echo json_encode([
            'leads' => $leads,
            'total' => $total,
            'page' => $page,
            'totalPages' => ceil($total / $limit)
        ]);
    }
    
    public static function addLeads($productId) {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $leadIds = $data['leadIds'] ?? [];
        
        if (empty($leadIds)) {
            http_response_code(400);
            echo json_encode(['error' => 'No lead IDs provided']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $added = 0;
        
        foreach ($leadIds as $leadId) {
            try {
                $stmt = $db->prepare("INSERT OR IGNORE INTO product_leads (product_id, lead_id) VALUES (?, ?)");
                $stmt->execute([$productId, $leadId]);
                if ($stmt->rowCount() > 0) $added++;
            } catch (Exception $e) {
                // Skip duplicates
            }
        }
        
        // Update product count
        $stmt = $db->prepare("UPDATE products SET count = (SELECT COUNT(*) FROM product_leads WHERE product_id = ?) WHERE id = ?");
        $stmt->execute([$productId, $productId]);
        
        echo json_encode(['added' => $added]);
    }
    
    public static function removeLeads($productId) {
        AuthController::requireAdmin();
        
        $data = json_decode(file_get_contents('php://input'), true);
        $leadIds = $data['leadIds'] ?? [];
        
        if (empty($leadIds)) {
            http_response_code(400);
            echo json_encode(['error' => 'No lead IDs provided']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $placeholders = implode(',', array_fill(0, count($leadIds), '?'));
        
        $stmt = $db->prepare("DELETE FROM product_leads WHERE product_id = ? AND lead_id IN ($placeholders)");
        $stmt->execute(array_merge([$productId], $leadIds));
        
        // Update product count
        $stmt = $db->prepare("UPDATE products SET count = (SELECT COUNT(*) FROM product_leads WHERE product_id = ?) WHERE id = ?");
        $stmt->execute([$productId, $productId]);
        
        echo json_encode(['removed' => count($leadIds)]);
    }
}
