<?php
class StripeController {
    
    public static function createCheckoutSession() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $stripeSecretKey = getenv('STRIPE_SECRET_KEY');
        if (!$stripeSecretKey) {
            http_response_code(500);
            echo json_encode(['error' => 'Stripe not configured']);
            return;
        }
        
        $packageId = $data['packageId'] ?? null;
        $email = $data['email'] ?? null;
        
        if (!$packageId) {
            http_response_code(400);
            echo json_encode(['error' => 'Package ID required']);
            return;
        }
        
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM credit_packages WHERE id = ?");
        $stmt->execute([$packageId]);
        $package = $stmt->fetch();
        
        if (!$package) {
            http_response_code(404);
            echo json_encode(['error' => 'Package not found']);
            return;
        }
        
        // Get or create pending purchase for non-logged-in users
        $userId = $_SESSION['user_id'] ?? null;
        $pendingPurchaseId = null;
        
        if (!$userId && $email) {
            $pendingPurchaseId = bin2hex(random_bytes(16));
            $stmt = $db->prepare("
                INSERT INTO pending_purchases (id, email, credits, amount, created_at, expires_at)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $pendingPurchaseId,
                $email,
                $package['credits'],
                $package['price'],
                time(),
                time() + 86400 // 24 hours
            ]);
        }
        
        // Create Stripe checkout session
        $priceInCents = (int)((float)$package['price'] * 100);
        
        $postData = http_build_query([
            'payment_method_types[0]' => 'card',
            'line_items[0][price_data][currency]' => 'cad',
            'line_items[0][price_data][product_data][name]' => $package['name'],
            'line_items[0][price_data][product_data][description]' => $package['credits'] . ' credits',
            'line_items[0][price_data][unit_amount]' => $priceInCents,
            'line_items[0][quantity]' => 1,
            'mode' => 'payment',
            'success_url' => 'http://' . $_SERVER['HTTP_HOST'] . '/checkout-success?session_id={CHECKOUT_SESSION_ID}' . ($pendingPurchaseId ? '&pending=' . $pendingPurchaseId : ''),
            'cancel_url' => 'http://' . $_SERVER['HTTP_HOST'] . '/buy-credits',
            'metadata[userId]' => $userId ?? '',
            'metadata[packageId]' => $packageId,
            'metadata[credits]' => $package['credits'],
            'metadata[pendingPurchaseId]' => $pendingPurchaseId ?? ''
        ]);
        
        if ($email) {
            $postData .= '&customer_email=' . urlencode($email);
        }
        
        $ch = curl_init('https://api.stripe.com/v1/checkout/sessions');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $stripeSecretKey
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to create checkout session']);
            return;
        }
        
        $session = json_decode($response, true);
        
        // Update pending purchase with session ID
        if ($pendingPurchaseId) {
            $stmt = $db->prepare("UPDATE pending_purchases SET stripe_session_id = ? WHERE id = ?");
            $stmt->execute([$session['id'], $pendingPurchaseId]);
        }
        
        echo json_encode(['url' => $session['url']]);
    }
    
    public static function handleWebhook() {
        $payload = file_get_contents('php://input');
        $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
        $webhookSecret = getenv('STRIPE_WEBHOOK_SECRET');
        
        if (!$webhookSecret) {
            http_response_code(500);
            echo json_encode(['error' => 'Webhook secret not configured']);
            return;
        }
        
        // Verify webhook signature
        $elements = explode(',', $sigHeader);
        $timestamp = null;
        $signature = null;
        
        foreach ($elements as $element) {
            $parts = explode('=', $element, 2);
            if (count($parts) === 2) {
                if ($parts[0] === 't') $timestamp = $parts[1];
                if ($parts[0] === 'v1') $signature = $parts[1];
            }
        }
        
        if (!$timestamp || !$signature) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid signature']);
            return;
        }
        
        $signedPayload = $timestamp . '.' . $payload;
        $expectedSignature = hash_hmac('sha256', $signedPayload, $webhookSecret);
        
        if (!hash_equals($expectedSignature, $signature)) {
            http_response_code(400);
            echo json_encode(['error' => 'Signature verification failed']);
            return;
        }
        
        $event = json_decode($payload, true);
        
        if ($event['type'] === 'checkout.session.completed') {
            $session = $event['data']['object'];
            $metadata = $session['metadata'] ?? [];
            
            $userId = $metadata['userId'] ?? null;
            $credits = (int)($metadata['credits'] ?? 0);
            $pendingPurchaseId = $metadata['pendingPurchaseId'] ?? null;
            
            $db = Database::getInstance()->getConnection();
            
            if ($userId) {
                // Add credits to existing user
                $stmt = $db->prepare("UPDATE users SET credits = credits + ? WHERE id = ?");
                $stmt->execute([$credits, $userId]);
            } elseif ($pendingPurchaseId) {
                // Update pending purchase as paid
                $stmt = $db->prepare("UPDATE pending_purchases SET stripe_session_id = ? WHERE id = ?");
                $stmt->execute([$session['id'], $pendingPurchaseId]);
            }
        }
        
        echo json_encode(['received' => true]);
    }
}
