<?php
// Load environment variables from .env file if it exists
if (file_exists(__DIR__ . '/.env')) {
    $lines = file(__DIR__ . '/.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            putenv(trim($key) . '=' . trim($value));
        }
    }
}

header('Content-Type: application/json');

$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$allowedOrigins = [
    'http://localhost:5000',
    'http://localhost:3000',
    'https://' . ($_SERVER['HTTP_HOST'] ?? '')
];

if (in_array($origin, $allowedOrigins) || empty($origin)) {
    header('Access-Control-Allow-Origin: ' . ($origin ?: '*'));
}
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Credentials: true');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

ini_set('session.cookie_httponly', '1');
ini_set('session.cookie_secure', '0');
ini_set('session.cookie_samesite', 'Lax');
session_start();

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/controllers/AuthController.php';
require_once __DIR__ . '/controllers/ProductController.php';
require_once __DIR__ . '/controllers/LeadController.php';
require_once __DIR__ . '/controllers/CreditController.php';
require_once __DIR__ . '/controllers/AdminController.php';
require_once __DIR__ . '/controllers/SettingsController.php';
require_once __DIR__ . '/controllers/StripeController.php';

$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$method = $_SERVER['REQUEST_METHOD'];

// Remove /api prefix if present
$uri = preg_replace('#^/api#', '', $uri);

// Route the request
try {
    // Auth routes
    if (preg_match('#^/auth/register$#', $uri) && $method === 'POST') {
        AuthController::register();
    } elseif (preg_match('#^/auth/login$#', $uri) && $method === 'POST') {
        AuthController::login();
    } elseif (preg_match('#^/auth/logout$#', $uri) && $method === 'POST') {
        AuthController::logout();
    } elseif (preg_match('#^/auth/user$#', $uri) && $method === 'GET') {
        AuthController::getUser();
    }
    
    // Product routes
    elseif (preg_match('#^/products$#', $uri) && $method === 'GET') {
        ProductController::getAll();
    } elseif (preg_match('#^/products/(\d+)$#', $uri, $matches) && $method === 'GET') {
        ProductController::getOne($matches[1]);
    } elseif (preg_match('#^/products$#', $uri) && $method === 'POST') {
        ProductController::create();
    } elseif (preg_match('#^/products/(\d+)$#', $uri, $matches) && $method === 'PUT') {
        ProductController::update($matches[1]);
    } elseif (preg_match('#^/products/(\d+)$#', $uri, $matches) && $method === 'DELETE') {
        ProductController::delete($matches[1]);
    } elseif (preg_match('#^/products/(\d+)/leads$#', $uri, $matches) && $method === 'GET') {
        ProductController::getLeads($matches[1]);
    } elseif (preg_match('#^/products/(\d+)/leads$#', $uri, $matches) && $method === 'POST') {
        ProductController::addLeads($matches[1]);
    } elseif (preg_match('#^/products/(\d+)/leads$#', $uri, $matches) && $method === 'DELETE') {
        ProductController::removeLeads($matches[1]);
    }
    
    // Lead routes
    elseif (preg_match('#^/leads$#', $uri) && $method === 'GET') {
        LeadController::getAll();
    } elseif (preg_match('#^/leads/(\d+)$#', $uri, $matches) && $method === 'GET') {
        LeadController::getOne($matches[1]);
    } elseif (preg_match('#^/leads$#', $uri) && $method === 'POST') {
        LeadController::create();
    } elseif (preg_match('#^/leads/(\d+)$#', $uri, $matches) && $method === 'PUT') {
        LeadController::update($matches[1]);
    } elseif (preg_match('#^/leads/(\d+)$#', $uri, $matches) && $method === 'DELETE') {
        LeadController::delete($matches[1]);
    } elseif (preg_match('#^/leads/bulk-delete$#', $uri) && $method === 'POST') {
        LeadController::bulkDelete();
    } elseif (preg_match('#^/leads/(\d+)/products$#', $uri, $matches) && $method === 'GET') {
        LeadController::getProducts($matches[1]);
    } elseif (preg_match('#^/leads/filter-options$#', $uri) && $method === 'GET') {
        LeadController::getFilterOptions();
    } elseif (preg_match('#^/leads/search$#', $uri) && $method === 'POST') {
        LeadController::search();
    }
    
    // Credit routes
    elseif (preg_match('#^/credit-packages$#', $uri) && $method === 'GET') {
        CreditController::getPackages();
    } elseif (preg_match('#^/credit-packages$#', $uri) && $method === 'POST') {
        CreditController::createPackage();
    } elseif (preg_match('#^/credit-packages/(\d+)$#', $uri, $matches) && $method === 'PUT') {
        CreditController::updatePackage($matches[1]);
    } elseif (preg_match('#^/credit-packages/(\d+)$#', $uri, $matches) && $method === 'DELETE') {
        CreditController::deletePackage($matches[1]);
    }
    
    // Saved searches
    elseif (preg_match('#^/saved-searches$#', $uri) && $method === 'GET') {
        LeadController::getSavedSearches();
    } elseif (preg_match('#^/saved-searches$#', $uri) && $method === 'POST') {
        LeadController::saveSearch();
    } elseif (preg_match('#^/saved-searches/(\d+)$#', $uri, $matches) && $method === 'DELETE') {
        LeadController::deleteSavedSearch($matches[1]);
    }
    
    // Purchased searches
    elseif (preg_match('#^/purchased-searches$#', $uri) && $method === 'GET') {
        LeadController::getPurchasedSearches();
    } elseif (preg_match('#^/purchased-searches$#', $uri) && $method === 'POST') {
        LeadController::purchaseSearch();
    } elseif (preg_match('#^/purchased-searches/(\d+)/export$#', $uri, $matches) && $method === 'GET') {
        LeadController::exportPurchasedSearch($matches[1]);
    }
    
    // Admin routes
    elseif (preg_match('#^/admin/upload$#', $uri) && $method === 'POST') {
        AdminController::uploadCsv();
    } elseif (preg_match('#^/admin/users$#', $uri) && $method === 'GET') {
        AdminController::getUsers();
    } elseif (preg_match('#^/admin/users/(\w+)$#', $uri, $matches) && $method === 'PUT') {
        AdminController::updateUser($matches[1]);
    } elseif (preg_match('#^/admin/stats$#', $uri) && $method === 'GET') {
        AdminController::getStats();
    } elseif (preg_match('#^/admin/enrich$#', $uri) && $method === 'POST') {
        AdminController::enrichLead();
    }
    
    // Apollo routes
    elseif (preg_match('#^/apollo/status$#', $uri) && $method === 'GET') {
        AdminController::apolloStatus();
    } elseif (preg_match('#^/apollo/search$#', $uri) && $method === 'POST') {
        AdminController::apolloSearch();
    }
    
    // Settings routes
    elseif (preg_match('#^/settings$#', $uri) && $method === 'GET') {
        SettingsController::getAll();
    } elseif (preg_match('#^/settings$#', $uri) && $method === 'POST') {
        SettingsController::update();
    }
    
    // Stripe routes
    elseif (preg_match('#^/stripe/create-checkout-session$#', $uri) && $method === 'POST') {
        StripeController::createCheckoutSession();
    } elseif (preg_match('#^/stripe/webhook$#', $uri) && $method === 'POST') {
        StripeController::handleWebhook();
    }
    
    // Not found
    else {
        http_response_code(404);
        echo json_encode(['error' => 'Not found', 'uri' => $uri, 'method' => $method]);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
